// Copyright 2014 Google Inc. All Rights Reserved.

#ifndef ANDROID_AUTO_PROJECTION_PROTOCOL_SENSOR_SOURCE_H
#define ANDROID_AUTO_PROJECTION_PROTOCOL_SENSOR_SOURCE_H

#include <bitset>
#include <set>

#include "common.h"
#include "ProtocolEndpointBase.h"

#define SENSOR_UPDATE_PERIOD_NEVER (-1)

/**
 * Implements the endpoint that is used to send sensor data from the car to the phone. Calls
 * to report data are automatically batched or dropped according to the parameters set up by
 * phone so the car can always be calling the report functions, the library will take care of
 * sending the data at the appropriate time.
 * A typical startup sequence would look like this:
 * <br>
 * <pre>
 *      galReceiver->init();
 *      ... Initialization code ...
 *      SensorSource* endpoint =
 *              new SensorSource(serviceId, galReceiver->messageRouter(), 0);
 *      endpoint->registerSensor(...); // Call for each sensor.
 *      endpoint->registerSensor(...); // Call for each sensor.
 *      endpoint->registerSensor(...); // Call for each sensor.
 *      galReceiver->registerService(endpoint);
 *      ... Other Initialization code ...
 *      galReceiver->start();
 * </pre>
 *
 * The list of mandatory and optional sensors can be found in the integration guide. This section
 * documents all possible sensors.
 */
class SensorSource : public ProtocolEndpointBase {
public:
    /**
     * @param id the service id.
     * @param router the MessageRouter to use.
     * @param locationCharacterization a bitmask of LocationCharacterization flags.
     * @param fuelTypes a set of all fuel types supported by this car.
     * @param evConnectorTypes a list of connector types supported for electric charging
     *     of this car, sorted from most preferred connector to least preferred. If this list is
     *     non-empty, the fuelTypes set must contain FUEL_TYPE_ELECTRIC element.
     */
    SensorSource(uint8_t id, MessageRouter *router, uint32_t locationCharacterization,
            const std::set<FuelType>& fuelTypes,
            const std::vector<EvConnectorType>& evConnectorTypes)
        : ProtocolEndpointBase(id, router, false),
            mLocationCharacterization(locationCharacterization),
            mLocationUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mCompassUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mSpeedUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mRpmUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mOdometerUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mFuelUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mParkingBrakeUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mGearUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mDiagnosticCodeUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mNightModeUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mEnvironmentDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mHvacDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mDrivingStatusDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mDeadReckoningDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mPassengerDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mDoorDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mLightDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mTirePressureDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mAccelerometerDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mGyroscopeDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mGpsSatelliteUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER),
            mTollCardDataUpdatePeriod(SENSOR_UPDATE_PERIOD_NEVER) {
        mFuelTypes = fuelTypes;
        if (!evConnectorTypes.empty()) {
            assert(mFuelTypes.find(FUEL_TYPE_ELECTRIC) != mFuelTypes.end());
        }
        mEvConnectorTypes = evConnectorTypes;
    }
    void addDiscoveryInfo(ServiceDiscoveryResponse *sdr);
    int routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg);
    void tick(uint64_t timestamp);

    /**
     * Use this method to report a GPS location. The eX multiplication is to allow for
     * fixed point representation of decimal values using an int32. For example, the value
     * 3.1415 becomes 31415000 in E7 notation and can be represented as an integer.
     * @param timestamp This parameter is deprecated; pass 0 when invoking.
     * @param latitudeE7 Latitude value [-90.0, +90.0] multiplied by 1e7.
     * @param longitudeE7 Longitude value [-180.0, +180.0) multiplied by 1e7.
     * @param hasAccuracy True if accuracy of this location is known and accuracyE3 param is valid.
     * @param accuracyE3 Horizontal 68% radius meters value multiplied by 1e3.
     * @param hasAltitude True if the altitudeE2 param should be considered, false otherwise.
     * @param altitudeE2 The altitude in meters above sea level multipled by 1e2.
     * @param hasSpeed True if the speedE3 param should be considered, false otherwise.
     * @param speedE3 The speed in m/s absolute velocity multiplied by 1e3.
     * @param hasBearing True if the bearingE6 param should be considered, false otherwise.
     * @param bearingE6 The compass bearing [0, 360) multiplied by 1e6.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportLocationData(uint64_t timestamp, int32_t latitudeE7, int32_t longitudeE7,
            bool hasAccuracy, int32_t accuracyE3, bool hasAltitude, int32_t altitudeE2,
            bool hasSpeed, int32_t speedE3, bool hasBearing, int32_t bearingE6);
    /**
     * Use this function to report the current value of an on-board magnetic compass. The value
     * reported here might be used in dead reckoning the position of the vehicle in the event
     * of a GPS signal loss.
     * @param bearingE6 The compass bearing [0, 360) multiplied by 1e6.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportCompassData(int32_t bearingE6);
    /**
     * Use this function to report the current car's orientation. Values reported here might be
     * used in dead reckoning the position of the vehicle in the event of a GPS signal loss.
     * @param bearingE6 The compass bearing in degrees from range [0, 360) multiplied by 1e6.
     * @param hasPitch True if pitch parameter provided is valid.
     * @param pitchE6 Car's pitch [-90, 90] multiplied by 1e6. Nose down is positive.
     * @param hasRoll True if roll parameter provided is valid.
     * @param rollE6 Car's roll (-180, 180] multiplied by 1e6. Right door down is positive.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportCompassData3D(int32_t bearingE6, bool hasPitch, int32_t pitchE6,
                            bool hasRoll, int32_t rollE6);
    /**
     * Use this function to report the current speed of the vehicle. The value reported here might
     * be used in dead reckoning the position of the vehicle in the event of a GPS signal loss.
     * @param speedE3 The speed in m/s signed velocity multiplied by 1e3.
     * @param hasCruiseEngaged True if cruise engaged paramater is valid.
     * @param cruiseEngaged Whether or not cruise controll is engaged.
     * @param hasCruiseSetSpeed True if cruise set speed paramter is valid.
     * @param cruiseSetSpeed The speed the cruise control is set at, in m/s.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportSpeedData(int32_t speedE3, bool hasCruiseEngaged, bool cruiseEngaged,
            bool hasCruiseSetSpeed, int32_t cruiseSetSpeed);
    /**
     * Use this function to report the current engine RPM value.
     * @param rpmE3 The engine RPM value multiplied by 1e3.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportRpmData(int32_t rpmE3);
    /**
     * Use this function to report the current value of the odometer.
     * @param kmsE1 The odometer data in kilometers multiplied by 1e3.
     * @param hasTripKms True if the trip kms paramter is valid.
     * @param tripKmsE1 Distance travelled since ignition was turned on multiplied by 1e1, in km.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportOdometerData(int32_t kmsE1, bool hasTripKms, int32_t tripKmsE1);
    /**
     * Use this function to report the amount of fuel remaining. This may be used as supplementary
     * input to navigation applications.
     * @param hasFuelRemainingPercent True if the fuelRemainingPercent parameter should be
     *        considered, false otherwise.
     * @param fuelRemainingPercent Fuel remaining in whole number percent values.
     * @param hasRangeKm True if the rangeKm parameter should be considered, false otherwise.
     * @param rangeKm The estimated remaining range for the current amount of fuel.
     * @param hasLowFuelWarning True if the low fuel warning parameter is valid.
     * @param lowFuelWarning True if the low fuel warning is on, false otherwise.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportFuelData(bool hasFuelRemainingPercent, int32_t fuelRemainingPercent,
            bool hasRangeKm, int32_t rangeKm, bool hasLowFuelWarning, bool lowFuelWarning);
    /**
     * Use this to report whether the parking brake is engaged or not. The value of this
     * sensor may be used to determine which UI elements can be interacted with.
     * @param engaged True if the parking brake is engaged, false otherwise.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportParkingBrakeData(bool engaged);
    /**
     * Use this method to report which gear the vehicle is in. The value of this sensor
     * may be used to determine which UI elements can be interacted with and which ones
     * get locked out. Additionally, these values might be used informationally.
     * @param gear Can be one of:
     *        GEAR_NEUTRAL,
     *        GEAR_1 .. GEAR_6,
     *        GEAR_DRIVE,
     *        GEAR_PARK,
     *        GEAR_REVERSE
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportGearData(int gear);
    /**
     * Use this to report any diagnostics data.
     * @param data The raw diagnostics data buffer. This is passed through without being
     *        interpreted.
     * @param len The size of the data buffer.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportDiagnosticsData(uint8_t *data, size_t len);
    /**
     * Use this to report the value of the day-night sensor. The value of this sensor will
     * affect the UI of projected applications.
     * @param night_mode true if night mode is enabled, false otherwise.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportNightModeData(bool night_mode);
    /**
     * Use this call to report data about the external environment.
     * @param hasTemperature Is the temperature value valid.
     * @param temperatureE3 The temperature in celcius multipled by 1e3.
     * @param hasPressure Is the pressure parameter value valid.
     * @param pressureE3 Pressure in kPA multiplied by 1e3.
     * @param hasRain True if the rain parameter is valid.
     * @param rain The rain detection level. 0 means no rain.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportEnvironmentData(bool hasTemperature, int32_t temperatureE3,
            bool hasPressure, int32_t pressureE3, bool hasRain, int32_t rain);
    /**
     * Use this call to report data about the HVAC system.
     * @param hasTargetTemperature Is the target temperature parameter valid.
     * @param targetTemperatureE3 Target temperature in celcius multipled by 1e3.
     * @param hasCurrentTemperature Is the cabin temperature param valid.
     * @param currentTemperatureE3 Current cabin temperature in celcius multipled by 1e3.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportHvacData(bool hasTargetTemperature, int32_t targetTemperatureE3,
            bool hasCurrentTemperature, int32_t currentTemperatureE3);
    /**
     * Use this call to report driving status change.
     * @param drivingStatus A bitmask of restrictions currently in effect. The available
     *        options are:
     *        DRIVE_STATUS_UNRESTRICTED,
     *        DRIVE_STATUS_NO_VIDEO,
     *        DRIVE_STATUS_NO_KEYBOARD_INPUT,
     *        DRIVE_STATUS_NO_VOICE_INPUT,
     *        DRIVE_STATUS_NO_CONFIG,
     *        DRIVE_STATUS_LIMIT_MESSAGE_LEN
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportDrivingStatusData(int32_t drivingStatus);
    /**
     * Use this call to report data about wheel speed and steering angle.
     * @param hasSteeringAngle Is the steeringAngleE1 param valid.
     * @param steeringAngleE1 Steering angle in tenths of a degree multiplied by 10.
     *        Left is negative.
     * @param wheelSpeedE3 An array of wheel speeds in m/s multiplied by 1e3. Start from the
     *        front left and go clockwise.
     * @param wheelSpeedEntries Number of elements in the wheel speed array.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportDeadReckoningData(bool hasSteeringAngle, int32_t steeringAngleE1,
            const int32_t* wheelSpeedE3, int32_t wheelSpeedEntries);
    /**
     * Use this call to report data about passenger presence.
     * @param passengerPresent true if a passenger is present, false otherwise.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportPassengerData(bool passengerPresent);
    /**
     * Use this call to report data about opened or closed doors.
     * @param hasHoodOpen Is the hoodOpen param valid.
     * @param hoodOpen True if the hood is open, false otherwise.
     * @param hasTrunkOpen Is the trunkOpen param valid.
     * @param trunkOpen True if the trunk is open, false otherwise.
     * @param doorOpen Array of door states, true if the door is open, false otherwise.
     * @param doorOpenEntries Number of elements in the door open array.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportDoorData(bool hasHoodOpen, bool hoodOpen,
            bool hasTrunkOpen, bool trunkOpen, const bool* doorOpen, int32_t doorOpenEntries);
    /**
     * Is this call to report data about headlights, turn signals etc.
     * @param hasHeadLightState Is the headLightState param valid.
     * @param headLightState The state of the head lights.
     * @param hasTurnIndicatorState Is the turnIndicatorState param valid.
     * @param turnIndicatorState The turn indicator state.
     * @param hasHazardLightsOn Is the hazardLightsOn param valid.
     * @param hazardLightsOn true if hazard lights are on, false otherwise.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportLightData(bool hasHeadLightState, int headLightState,
            bool hasTurnIndicatorState, int turnIndicatorState,
            bool hasHazardLightsOn, bool hazardLightsOn);
    /**
     * Use this call to report data about tire pressures.
     * @param tirePressures An array of tire pressure values in psi multiplied by 1e2.
     * @param numEntries The number of elements in the tirePressuresE2 array.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportTirePressureData(const int32_t* tirePressuresE2, int32_t numEntries);
    /**
     * Use this call to report data from accelerometer (include gravity). Units are m/s^2 multiplied
     * by 1e3.
     * @param hasAccelerationX Acceleration along X-axis is valid.
     * @param xAccelerationE3 Acceleration from left door to right.
     * @param hasAccelerationY Acceleration along Y-axis is valid.
     * @param yAccelerationE3 Acceleration from back to nose.
     * @param hasAccelerationZ Acceleration along Z-axis is valid.
     * @param zAccelerationE3 Acceleration from floor to ceiling.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportAccelerometerData(bool hasAccelerationX, int32_t xAccelerationE3,
            bool hasAccelerationY, int32_t yAccelerationE3, bool hasAccelerationZ,
            int32_t zAccelerationE3);
    /**
     * Use this call to report data from gyroscope. Units are rad/s multiplied by 1e3.
     * @param hasRotationSpeedX Rotation speed around X-axis is valid.
     * @param xRotationSpeedE3 Rotation speed around axis from left door to right.
     * @param hasRotationSpeedX Rotation speed around X-axis is valid.
     * @param yRotationSpeedE3 Rotation speed around axis from back to nose.
     * @param hasRotationSpeedX Rotation speed around X-axis is valid.
     * @param zRotationSpeedE3 Rotation speed around axis from floor to ceiling.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportGyroscopeData(bool hasRotationSpeedX, int32_t xRotationSpeedE3,
            bool hasRotationSpeedY, int32_t yRotationSpeedE3, bool hasRotationSpeedZ,
            int32_t zRotationSpeedE3);
    /**
     * Use this call to report satellite data from GPS. At least number in-use must be avaialble.
     * If per-satellite info is available then arrays must contain {@code numberInView} elements.
     * @param numberInUse Number of satellites used in GPS fix.
     * @param hasNumberInView Whether {@code numberInView} is valid.
     * @param numberInView Number of satellites visible to the GPS receiver.
     * @param prns Array of PRNs of satellites in view or NULL if per-satellite info unavailable.
     * @param snrsE3 Array of SNRs of satellites in view in dB multiplied by 1e3 or NULL
     *        if per-satellite info unavailable.
     * @param usedInFix Array of flags whether this satellite was used in GPS fix or NULL
     *        if per-satellite info unavailable.
     * @param azimuthsE3 Array of azimuths of satellites in degrees clockwise from north
     *        multiplied by 1e3 or NULL if per-satellite info unavailable or position data for
     *        satellites is absent.
     * @param elevationsE3 Array of elevations of satellites in degrees from horizon to zenith
     *        multiplied by 1e3 or NULL if per-satellite info unavailable or position data for
     *        satellites is absent.
     * @return STATUS_SUCCESS on success, an error code otherwise.
     */
    int reportGpsSatelliteData(int32_t numberInUse, bool hasNumberInView, int32_t numberInView,
            const int32_t* prns, const int32_t* snrsE3, const bool* usedInFix,
            const int32_t* azimuthsE3, const int32_t* elevationsE3);
    /**
     * Use this call to report data about the Electronic Toll Card.
     * @param isCardPresent True if the Electronic Toll Card is present, false
     * otherwise.
     */
    int reportTollCardData(bool isCardPresent);

    /**
     * Use this to register a sensor to be advertised during service discovery.
     * Must be called during initialization.
     * @param type The type of the sensor being registered. This should be one
     * of: SENSOR_LOCATION, SENSOR_COMPASS, SENSOR_SPEED, SENSOR_RPM,
     *        SENSOR_ODOMETER,
     *        SENSOR_FUEL,
     *        SENSOR_PARKING_BRAKE,
     *        SENSOR_GEAR,
     *        SENSOR_OBDII_DIAGNOSTIC_CODE,
     *        SENSOR_NIGHT_MODE,
     *        SENSOR_ENVIRONMENT_DATA,
     *        SENSOR_HVAC_DATA,
     *        SENSOR_DRIVING_STATUS_DATA,
     *        SENSOR_DEAD_RECKONING_DATA,
     *        SENSOR_PASSENGER_DATA,
     *        SENSOR_DOOR_DATA,
     *        SENSOR_LIGHT_DATA,
     *        SENSOR_TIRE_PRESSURE_DATA,
     *        SENSOR_ACCELEROMETER_DATA,
     *        SENSOR_GYROSCOPE_DATA,
     *        SENSOR_GPS_SATELLITE_DATA,
     *        SENSOR_TOLL_CARD
     */
    void registerSensor(int type) { mSensors.insert(type); }

    /**
     *  Report transient or permanent sensor error to MD. This is also used to report recovery
     *  from such error state by setting errorType to SENSOR_OK.
     *  Send this message when HU cannot recover from error immediately. If error is recoverable
     *  immediately, there is no need to send this message.
     *  @param type the type of sensor having problem. This can be SENSOR_* like SENSOR_LOCATION.
     *  @param errorType This should be one of:
     *         SENSOR_ERROR_OK
     *         SENSOR_ERROR_TRANSIENT
     *         SENSOR_ERROR_PERMANENT
     */
    int reportSensorError(int type, int errorType);

private:
    int handleSensorRequest(const SensorRequest& req);
    void sendSensorBatch(const SensorBatch& batch);
    void sendSensorResponse(int32_t status);
    void populateFuelData(FuelData* data, bool hasFuelRemainingPercent,
            int32_t fuelRemainingPercent, bool hasRangeKm, int32_t rangeKm,
            bool hasLowFuelWarning, bool lowFuelWarning);

    std::set<int32_t> mSensors;
    uint32_t mLocationCharacterization;
    std::set<FuelType> mFuelTypes;
    std::vector<EvConnectorType> mEvConnectorTypes;
    int64_t mLocationUpdatePeriod;
    int64_t mCompassUpdatePeriod;
    int64_t mSpeedUpdatePeriod;
    int64_t mRpmUpdatePeriod;
    int64_t mOdometerUpdatePeriod;
    int64_t mFuelUpdatePeriod;
    int64_t mParkingBrakeUpdatePeriod;
    int64_t mGearUpdatePeriod;
    int64_t mDiagnosticCodeUpdatePeriod;
    int64_t mNightModeUpdatePeriod;
    int64_t mEnvironmentDataUpdatePeriod;
    int64_t mHvacDataUpdatePeriod;
    int64_t mDrivingStatusDataUpdatePeriod;
    int64_t mDeadReckoningDataUpdatePeriod;
    int64_t mPassengerDataUpdatePeriod;
    int64_t mDoorDataUpdatePeriod;
    int64_t mLightDataUpdatePeriod;
    int64_t mTirePressureDataUpdatePeriod;
    int64_t mAccelerometerDataUpdatePeriod;
    int64_t mGyroscopeDataUpdatePeriod;
    int64_t mGpsSatelliteUpdatePeriod;
    int64_t mTollCardDataUpdatePeriod;
    SensorBatch mLastValues;
    std::bitset<SensorType_ARRAYSIZE> mLastValuesValid;
    GalMutex mCacheLock;
};

#endif // ANDROID_AUTO_PROJECTION_PROTOCOL_SENSOR_SOURCE_H
